const fs = require('fs');
const path = require('path');
const https = require('https');
const express = require('express');
const socketIo = require('socket.io');

// ============ CONFIG ============
const CPANEL_USER = 'dropapp';
const DOMAIN = `socket.${CPANEL_USER}.dev`;
const PRIVATE_DIR = `/home/${CPANEL_USER}/ssl/keys/`;
const CERT_DIR = `/home/${CPANEL_USER}/ssl/certs/`;
const PORT = 4949;

// ============ HELPERS ============
// Convert domain dots to underscores
function domainToCpanelPrefix(domain) {
  return domain.replace(/\./g, '_');
}

// Get newest file containing prefix anywhere in filename
function getNewestFileByContains(dir, prefix) {
  try {
    const files = fs.readdirSync(dir)
      .filter(name => name.includes(prefix))
      .map(name => ({ name, mtime: fs.statSync(path.join(dir, name)).mtime.getTime() }))
      .sort((a, b) => b.mtime - a.mtime);

    return files.length ? path.join(dir, files[0].name) : null;
  } catch (err) {
    return null;
  }
}

// Find CA bundle similarly
function findCABundle(dir, prefix) {
  const patterns = [
    '.ca-bundle',
    '.ca_bundle',
    '.cabundle',
    '-ca-bundle',
    '-ca_bundle',
  ];

  try {
    const files = fs.readdirSync(dir).filter(name => name.includes(prefix));
    for (const ptn of patterns) {
      const found = files.find(f => f.includes(ptn));
      if (found) return path.join(dir, found);
    }
  } catch (err) {
    return null;
  }
  return null;
}

// ============ LOCATE SSL FILES ============
const prefix = domainToCpanelPrefix(DOMAIN);

const latestKey = getNewestFileByContains(PRIVATE_DIR, prefix);
const latestCert = getNewestFileByContains(CERT_DIR, prefix);
const latestCA = findCABundle(CERT_DIR, prefix);

if (!latestKey || !latestCert) {
  console.error('❌ Could not find SSL files for domain:', DOMAIN);
  console.error('Searched in:', PRIVATE_DIR, CERT_DIR);
  process.exit(1);
}

console.log('Using key:', latestKey);
console.log('Using cert:', latestCert);
if (latestCA) console.log('Using CA bundle:', latestCA);

// ============ SSL OPTIONS ============
const sslOptions = {
  key: fs.readFileSync(latestKey),
  cert: fs.readFileSync(latestCert),
  ...(latestCA && { ca: fs.readFileSync(latestCA) }),
};

// ============ EXPRESS + HTTPS ============
const app = express();
app.use(express.static(path.join(__dirname, 'public')));

const httpsServer = https.createServer(sslOptions, app);

// ============ SOCKET.IO ============
const io = socketIo(httpsServer, {
  cors: { origin: '*' },
});

io.of('/').on('connection', (socket) => {
  console.log('client connected:', socket.id);

  socket.on('sendOrder', (data) => {
    const roomName = data[0];
    console.log('sendOrder ->', roomName);
    socket.emit('sendOrder', 'roomName');
    socket.broadcast.to(roomName).emit('receiveOrder', [data[1], data[2]]);
  });

  socket.on('sendUpdate', (data) => {
    const roomName = data[0];
    console.log('sendUpdate ->', roomName);
    socket.emit('sendOrder', 'roomName');
    socket.broadcast.to(roomName).emit('receiveOrderUpdate', [data[1], data[2]]);
  });

  socket.on('showMsg', (data) => {
    const roomName = data[0];
    console.log('showMsg ->', roomName);
    socket.emit('showMsg', 'roomName');
    socket.broadcast.to(roomName).emit('showMsg', [data[1]]);
  });

  socket.on('sendPrint', (data) => {
    const roomName = data[0];
    console.log('sendPrint ->', roomName);
    socket.emit('sendPrint', 'roomName');
    socket.broadcast.to(roomName).emit('receivePrint', [data[1], data[2]]);
  });

  socket.on('sendRefresh', (data) => {
    const roomName = data[0];
    console.log('sendRefresh ->', roomName);
    socket.broadcast.to(roomName).emit('sendRefresh', data[1]);
  });

  socket.on('joinRoom', (roomName) => {
    console.log('joinRoom ->', roomName);
    const lastRoom = Array.from(socket.rooms)[1];
    if (lastRoom) socket.leave(lastRoom);
    socket.join(roomName);
    socket.emit('joinRoom', roomName);
  });

  socket.on('disconnect', () => {
    console.log('client disconnected:', socket.id);
  });
});

// ============ START ============
httpsServer.listen(PORT, () => {
  console.log(`🚀 Secure Socket.IO server running at https://${DOMAIN}:${PORT}`);
});
